/*
 * This file is part of John the Ripper password cracker.
 *
 * Using md5_std.c as a starting point, make version for OpenVMS password
 * hash.  This version uses Posix threads to perform the hashing in
 * parallel. Compile with macro HAVE_PTHREADS defined to enable multi-threading,
 * otherwise threading will be disabled.
 *
 * Environment variables examined at initialization:
 *    JOHN_OPENVMS_THREADS  Dispatch parameters for threading, value is
 *                          of for "n m l" where:
 *                            n - Number of threads to create (default=1).
 *                            m - Min. batch size for nonblocked start (def=32).
 *                            l - Max. batch size for worker thread (def=60).
 *    JOHN_OPENVMS_DEBUG n  Output an exectution trace if non-zero.
 *
 * Limit and threshold value can have a maximum of the module SAVE_LIMIT
 * setting, which is 240.  If minimum batch size (m) is specified as *,
 * the code attempts to self tune the threshold (initially l/(n+1)) to
 * keep the worker threads at maximum utilization.
 *
 * Author:  David Jones
 * Date:    9-JUL-2009
 * Revised: 17-JUL-2009     Cleanup, make work under OS X.
 * Revised: 26-JUL-2009     Cleanup comments.
 * Revised: 31-JUL-2009     Fix uaf_qword reference for NOLONGLONG compiles.
 * Revised:  2-AUG-2009     Eliminate redundant update of hint table entries.
 * Revised: 27-AUG-2011     Remove log_event calls.
 * Revised: 9-SEP-2011      Rewrite dispatch_pending() for better
 *                          multi-threading.
 * Revised: 14-SEP-2011     Change password max length from 31 to 32.
 * Revised: 19-SEP-2011     Replace NOPTHREADS with HAVE_PTHREADS.
 * Revised: 24-SEP-2011	    Remove report_statistics() function.
 *
 * Copyright (c) 2011 by David L. Jones <jonesd/at/columbus.rr.com>, and
 * is hereby released to the general public under the following terms:
 *    Redistribution and use in source and binary forms, with or without
 *    modifications, are permitted.
 */
#include <string.h>
#include "arch.h"
#include "common.h"
#include "logger.h"

#ifdef HAVE_PTHREADS
#include <pthread.h>
#define MUTEX pthread_mutex_t
#define LOCK_MUTEX(lck) pthread_mutex_lock(lck)
#define UNLOCK_MUTEX(lck) pthread_mutex_unlock(lck)
#define INIT_MUTEX(lck) pthread_mutex_init(lck, 0)
#define CONDVAR pthread_cond_t
#define THREAD_ATTR pthread_attr_t
#else
#include <time.h>
#define MUTEX long
#define LOCK_MUTEX(lck)
#define UNLOCK_MUTEX(lck)
#define INIT_MUTEX(lck)
#define CONDVAR  long
#define THREAD_ATTR long
#endif

#ifdef VMS
#include <uaidef.h>
#include <descrip.h>
#include <lib$routines.h>
#include <times.h>
#else
#include <sys/times.h>
#endif
#ifndef UAI$M_PWDMIX
#define UAI$M_PWDMIX 0x2000000
#endif
#include "vms_std.h"            /* includes uaf_encode.h */

#ifndef TRY                     /* fallback if excpetions not support */
#define TRY if ( wq )
#define CATCH_ALL else
#define THIS_CATCH 0
#define RERAISE printf("No exception support")
#define ENDTRY
#define pthread_exc_report_np(x) printf("Exception %d caught\n",(int)(x))
#define pthread_exc_get_status_np(c,v) (*(v)=0)
#elif !defined(VMS)
#define pthread_exc_report_np(x) printf("Exepction %d caught\n", (int)(x))
#define pthread_exc_get_status_np(c,v) (*(v)=0)
#endif

/*
 * Static storage for data preserved across calls. Crypt_out and summary
 * are shared with VMS_FMT module (by way of global pointers).  
 * Summary structure is used by cmp_all to quickly reject non-matches.
 */
static struct result_hash crypt_out[SAVE_LIMIT];
static struct result_hint *summary;
struct result_hint *VMS_std_hint = 0;
struct result_hash *VMS_std_crypt_out = 0;

int VMS_dbg_flag;
#define STATS_INTERVAL 10000            /* Crypt calls between stats report */
#define DBGPR if ( VMS_dbg_flag == 0 ); else printf
#define DEFAULT_WORKER_COUNT "1"
#define DEFAULT_PENDING_THRESHOLD "24"
#define DEFAULT_PENDING_LIMIT "360"
#define PLAINTEXT_LENGTH 32
#define WO_STS_DEFUNCT 0
#define WO_STS_BUSY 1
#define WO_STS_COMPLETED 2

/*
 * Work_item struct saves arguments from a set_key request.
 */
struct work_item {
    int ndx;                      /* Index of output location in crypt_out */
    char key[PLAINTEXT_LENGTH+1]; /* candidate password */
    char pad[3];
};
/*
 * The work_queue structure collects set_key/set_salt requests and tracks
 * the assignment of blocks of pending work items to worker threads.  The
 * work_order structure coordinates this handoff been the main thread and
 * a single worker thread.
 */
struct work_queue {
    MUTEX lock;			/* Protects structures for shared access */
    CONDVAR order_done;         /* Signaled by worker at  BUSY -> COMPLETED */
    int worker_count;		/* number of threads (size of work_order[]) */
    int workers_active;		/* Number of worker threads with BUSY status */

    struct {			/* Statistics for VMS_std_crypt() calls */
	long calls;		/* Number of times crypt function called */
	long mile_marker;	/* Report stats when calls == mile_marker */
	long keys_computed;	/* Total hash computations requested */
	long preempts;		/* number of times dispatch_pending preempts */
	long preempt_idle;	/* Counts no active workers after preempt */
	clock_t start_clock, cur_clock;
	struct tms start_cpu, cur_cpu;
    } crypt_counters;
    struct uaf_hash_info salt;	/* Salt for new orders */
    int salt_valid;		/* true if salt set since last crypt */
    int keys_changed;		/* true if any keys set since last crypt */
    int request_count;		/* Requests pending at current sequence */
    int dispatched;		/* Number of pending  */
    int assigned;		/* number of pending assigned to workers */
    int pending_threshold;	/* Count at which to create work order */
    int pending_limit;		/* Max. count allowed in work order */
    struct work_order *order;	/* Array of wq->worker_count size */
    struct work_item request[SAVE_LIMIT];   /* Candidate passwords to encrypt */
};

struct work_order {
    struct work_queue *wq;
    int assignee;		/* thread number assigned to do work */
    int status;			/* 0-initial, 1-busy, 2-completed */
    CONDVAR order_ready;	/* Signaled when status changed to BUSY */

    struct uaf_hash_info salt;	/* Salt to use to process items */

    struct work_item *first;	/* Points into wq->request array. */
    int count;			/* Number of items to process */
    int stalls;			/* counts times work completed soon */
    char scratch[8140];
};
static struct work_queue *wq;
/*
 * Process request item[0..count-1] array, placing result of hashes in the 
 * crypt_out array.
 * Caller must hold held_lock mutex, which we release and re-acquire.
 */
static void compute_hashes ( int assignee, MUTEX *held_lock,
	struct uaf_hash_info *salt, struct work_item *item, int count )
{
    int ndx, match;
    uaf_lword smask, info;
    struct work_item *cur_item, *end_item;
    struct {				/* Array to save results for */
	int ndx;			/* inclusion in summary table */
	uaf_lword *seen_entry;		/* points into seen array */
    } *resptr, result[count+1];
    /*
     * Compute hash of each item[] password using the same input salt.  The 
     * test_password  function saves resulting hash back to the uaf_hash_info 
     * structure (1st argument) when the replace_if argument is 1 (true). 
     */
    UNLOCK_MUTEX ( held_lock );
/*printf ( "Worker %d computing %d starting at %x\n", assignee, count,item );*/
/*    if ( assignee ) pthread_yield_np(); */
    end_item = &item[count];		/* one past final item to process */
    resptr = result;			/* Saves result for insert in summary */
    smask = summary->mask;
    for ( cur_item = item; cur_item < end_item; cur_item++ ) {
	ndx = cur_item->ndx;		/* Output location in crypt_out[] */
	crypt_out[ndx].info = *salt;	/* set salt for hash */
	match = uaf_test_password (&crypt_out[ndx].info, cur_item->key, 1);
	resptr->ndx = ndx;
	resptr->seen_entry =
		&summary->seen[UAF_QW_AND(crypt_out[ndx].info.hash,smask)];
	resptr++;
    }
    /*
     * Update summary table using nxtseq value as generation number.
     * Table entries are relatively close, so keep summary->lock mutex entire
     * time.
     */
    DBGPR ( "/VMS_STD-%d/ updating %d hint table entries.\n", assignee, count );
    LOCK_MUTEX ( summary->lock_vp );
    if ( summary->nxtseq == 0 ) {
	/* We wrapped, reset table and start seq at 1 (zero never used)*/
	memset ( summary->seen, 0, (summary->mask+1)*sizeof(uaf_lword) );
	summary->nxtseq = 1;
    }
    while ( (--resptr) >= result ) {
	info = (*resptr->seen_entry) & HINT_GENERATION_MASK;
	ndx = resptr->ndx;
	if ( info == summary->nxtseq ) ndx = HINT_MULTIPLE;   /* double hit */
	*resptr->seen_entry = summary->nxtseq | (ndx<<HINT_GENMASK_BITS);
    }
    UNLOCK_MUTEX ( summary->lock_vp );
    LOCK_MUTEX ( held_lock );
}
#ifdef HAVE_PTHREADS
/*
 * Thread start routine for a worker thread.
 */
static void *worker_start ( void *work_order_vp )
{
    struct work_order *wo;		/* Work_order_vp as non void * */
    /*
     * Main loop.  Assume initial status of BUSY with zero work items, main
     * thread can't add items to the work order until we mark it completed.
     */
    wo = work_order_vp;
    LOCK_MUTEX ( &wo->wq->lock );
    TRY do {
	/*
	 * Hash the designated subset of wq->request (count items starting
	 * at wo->first).  Lock will be released and re-acquired.
	 */
	compute_hashes ( wo->assignee, &wo->wq->lock, 
		&wo->salt, wo->first, wo->count );
	/*
	 * Notify main thread via order_done condition that we finished.
	 */
	wo->wq->workers_active--;
	wo->status = WO_STS_COMPLETED;
	if ( 0 != pthread_cond_signal ( &wo->wq->order_done ) ) {
	    fprintf (stderr, "Worker start internal error, cond_signal\n");
	    return wo;
	}
	/*
	 * Wait for main thread to change work order's state back to BUSY, 
         * indicating it has updated it with new input arguments.
	 * Thread will be shut down if marked busy and assignee number cleared.
	 */
	wo->stalls++;
	do if ( 0 != pthread_cond_wait (&wo->order_ready, &wo->wq->lock) ) {
	    fprintf ( stderr, "Condition wait error, abort\n" );
	    UNLOCK_MUTEX ( &wo->wq->lock );
	    return wo;
	} while ( wo->status != WO_STS_BUSY );

    } while ( wo->assignee );
    CATCH_ALL {
	unsigned int stsval;
	printf ( "Exception in worker thread %d\n", wo->assignee );
	if ( pthread_exc_get_status_np ( THIS_CATCH, &stsval ) == 0 ) {
	    printf ( "Status code: %d\n", stsval );
	} else printf ( "Status retrieval failed\n" );
	pthread_exc_report_np ( THIS_CATCH );
	RERAISE;
    }
    ENDTRY
    UNLOCK_MUTEX ( &wo->wq->lock );
    return wo;
}
/*
 * Make one pass through worker list and assign work to idle workers found.
 * Number of requests assigned to each worker will be between min_lot and
 * max_lot size.  Return value is number on request list left unassigned.
 * We assume wq->lock is held by caller.
 */
static int assign_work ( int min_lot, int max_lot )
{
    int worker_ndx, lot_size;
    struct work_order *wo;
    /*
     * Loop through list of workers and look for idle.
     */
    for ( worker_ndx = 0; worker_ndx < wq->worker_count; worker_ndx++ ) {
	wo = &wq->order[worker_ndx];
	if ( wo->status == WO_STS_BUSY ) continue;
	/*
	 * Found free worker, maximize work to give it.
	 */
     	lot_size = wq->request_count - wq->assigned;
	if ( lot_size <= 0 ) return 0;		/* no work, bail out */
	if ( lot_size < min_lot ) return lot_size;
	if ( lot_size > max_lot ) lot_size = max_lot;
	/*
	 * Fill in work order details and hand off to worker thread.
	 */
	wo->status = WO_STS_BUSY;
	wo->first = &wq->request[wq->assigned];   /* First unassigned */
	wo->count = lot_size;
	wo->stalls = 0;
	wo->salt = wq->salt;

	pthread_cond_signal ( &wo->order_ready );
	wq->assigned += lot_size;
	wq->workers_active++;
	DBGPR ( "/VMS_STD/ initiated work order #%d, count: %d [%d:%d]\n",
	   wo->assignee, wo->count, min_lot, max_lot );
    }
    /*
     * We are left with more than min_lot requests pending and no available
     * workers.
     */
    return wq->request_count - wq->assigned;
}
#else
#define assign_work(m,n) (wq->request_count-wq->assigned)
#endif   /* HAVE_PTHREADS */
/*
 * Process work queue request items not yet dispatched to workers.
 */
static void dispatch_pending ( const char *caller, int wait_for_completion ) 
{
    int backlog, lot_size, idle_workers;
    /*
     * Mark all in list as 'dispatched' and assign as many as we can to
     * idle workers.  Backlog variable will hold number remaining to assign.
     */
    LOCK_MUTEX ( &wq->lock );
    idle_workers = wq->worker_count - wq->workers_active;
    wq->dispatched = wq->request_count;
    DBGPR ( "/VMS_STD/ %s called dispatch_pending, wait=%d, idle=%d, req=%d\n",
	caller, wait_for_completion, idle_workers, wq->request_count );

    if ( idle_workers > 0 ) {
	backlog = assign_work ( 0, wq->pending_limit );
    } else {
	backlog = wq->request_count - wq->assigned;
    }
    DBGPR ( "/VMS_STD/ backlog now %d\n", backlog );
    /*
     * Let backlog build up if asynchronous call.
     */
    if ( !wait_for_completion ) {
	UNLOCK_MUTEX ( &wq->lock );
	return;
    }
    /*
     * Synchronous call.  Do productive 'busy wait' until all pending
     * key requests have been assigned.
     */
    while ( backlog > 0 ) {
        /*
	 * Assign self (this thread) to do no more than pending_threshold keys,
         * then re-attempt to offload to idle workers.
	 */
	if ( backlog > wq->pending_threshold ) backlog = wq->pending_threshold;
	wq->crypt_counters.preempts++;
	compute_hashes ( 0, &wq->lock, &wq->salt, 
		&wq->request[wq->assigned], backlog );
	wq->assigned += backlog;
	if ( wq->workers_active == 0 ) wq->crypt_counters.preempt_idle++;
	
        if ( wq->worker_count > wq->workers_active ) {
	    lot_size = backlog/2;   /* we'll still do half */
	    if ( lot_size > wq->pending_limit ) lot_size = wq->pending_limit;
	    if ( lot_size < wq->pending_threshold) lot_size = wq->pending_threshold;
	    backlog = assign_work ( wq->pending_threshold, lot_size );
	} else backlog = wq->request_count - wq->assigned;
    }
    /*
     * All work assigned, wait for worker threads to finish.
     */
#ifdef HAVE_PTHREADS
    while ( wq->workers_active > 0 ) {
	if ( 0 != pthread_cond_wait(&wq->order_done, &wq->lock) ) {
	    fprintf(stderr,"Condition wait error, abort\n"); return;
	}
    }
#endif
    UNLOCK_MUTEX ( &wq->lock );

}
/*
 * Initialization function for module, called by main program.
 */
void VMS_std_init(void)
{
    char *env, *arg;
    int workers, threshold, limit, smask;
#ifdef HAVE_PTHREADS
    int i;
    static THREAD_ATTR attr;
#endif
    MUTEX *lock_p;
    /*
     * Read configuration from environment variables.
     */
    env = getenv ( "JOHN_OPENVMS_DEBUG" );		/* set debug mode */
    if ( env ) VMS_dbg_flag = atoi ( env );

    env = getenv ( "JOHN_OPENVMS_THREADS" );
    arg = env ? strtok(env," ") : (char *) 0;
    workers = atoi ( arg ?  arg : DEFAULT_WORKER_COUNT );
    if ( arg ) arg = strtok ( 0, " " );
    threshold = atoi ( arg  ? arg : DEFAULT_PENDING_THRESHOLD );
    if ( arg ) arg = strtok ( 0, " " );
    limit = atoi ( arg ? arg : DEFAULT_PENDING_LIMIT );
    /*
     * Initialize the structure for storing result hints (speeds up cmp_all).
     * Structure ends in seen array which is variably sized to be at least
     * 200 times larger than SAVE_LIMIT.  cmp_all should get a false positive
     * no more than 1% of the time.
     */
    lock_p = malloc ( sizeof(MUTEX) );
    for ( smask = 255; smask < (SAVE_LIMIT*200); ) smask = (smask<<1) | 1;
    summary = calloc(1, sizeof(struct result_hint)+(smask*sizeof(uaf_lword)));
    summary->mask = smask;
    VMS_std_hint = summary;
    /*
     * Initialize for multi-threading.  Create crypting threads, one for
     * each processor and assign a thread number 1..N.
     */
    uaf_init ( );
    summary->lock_vp = lock_p;		/* hint struct uses opaque pointer */
    INIT_MUTEX ( summary->lock_vp );

    wq = malloc(sizeof(struct work_queue));
    wq->worker_count = (workers > 0) ? workers : 0;
    wq->order = malloc(sizeof(struct work_order)*(wq->worker_count+1));
    wq->pending_limit = (limit < SAVE_LIMIT) ? limit : SAVE_LIMIT;
    wq->pending_threshold = (threshold < limit) ? threshold : limit;

    INIT_MUTEX ( &wq->lock );
#ifdef HAVE_PTHREADS
    pthread_cond_init ( &wq->order_done, 0 );
    pthread_attr_init ( &attr );
    pthread_attr_setdetachstate ( &attr, PTHREAD_CREATE_DETACHED );
    pthread_attr_setstacksize ( &attr, 
	PTHREAD_STACK_MIN + (SAVE_LIMIT*2*sizeof(uaf_lword *)) );

    for ( i = 0; i < wq->worker_count; i++ ) {
	pthread_t tid;
        memset ( &wq->order[i], 0, sizeof(struct work_order) );
	wq->order[i].wq = wq;	/* back link */
	wq->order[i].status = WO_STS_BUSY;   /* in_progress, worker completes */
	wq->order[i].assignee = i+1;
	wq->workers_active++;
	pthread_cond_init ( &wq->order[i].order_ready, 0 );

	if (0 != pthread_create ( &tid, &attr, worker_start, &wq->order[i])) {
	    fprintf ( stderr, "Error creating worker thread #%d\n", i+1 );
	}
    }
#else
    wq->worker_count = 0;		/* Force skipping thread scheduling */
#endif
    memset ( &wq->crypt_counters, 0, sizeof ( wq->crypt_counters ) );
}
/*
 * Save key (password) for producing ciphertext (hash) from it and saved salt.
 */
int VMS_std_set_key(char *key, int index)
{
    int unbusy, request_count;
    /*
     * Validate index argument in range and reset request_count if first
     * set_key after a crypt call.  This lets caller change the salt and
     * call crypt again to compute new hashes with the same set of saved keys.
     */
    if ( (index < 0) || (index >= SAVE_LIMIT) ) {
	fprintf ( stderr,"BUGCHECK, vms_std_set_key: index=%d out of range",index);
	return 0;
    }
    if ( !wq->keys_changed ) {
	wq->keys_changed = 1;
	wq->request_count = 0;
	wq->assigned = 0;
	wq->dispatched = 0;
    }
    request_count = wq->request_count;
    if ( request_count >= SAVE_LIMIT ) {
	printf ( "Bugcheck, too many set_keys, req=%d, limit=%d!\n",
		request_count, SAVE_LIMIT );
	request_count = SAVE_LIMIT-1;
    }
    /*
     * Add key and index (output location) to list of pending items.
     */
    wq->request[request_count].ndx = index;
    if ( strlen(key) > PLAINTEXT_LENGTH ) {
	memcpy ( wq->request[request_count].key, key, PLAINTEXT_LENGTH );
        wq->request[request_count].key[PLAINTEXT_LENGTH] = 0;
    } else strcpy ( wq->request[request_count].key, key );
    DBGPR ( "/VMS_STD/ saved key[%d] as item[%d] '%s'\n", index,
	request_count, wq->request[request_count].key );
    wq->request_count = request_count+1;
    /*
     * If a salt has been set since last crypt operation, see if we should
     * start clearing the list with another thread.
     */
    if ( wq->salt_valid ) {
	unbusy = wq->request_count - wq->dispatched;
	if ( unbusy >= wq->pending_threshold ) dispatch_pending ("set key", 0);
    }
    return request_count;	/* Position key saved at in wq->request array */
}
/*
 * Save salt for producing ciphertext from it and saved keys at next crypt call.
 */
void VMS_std_set_salt ( void *salt_vp )
{
    int pending;
    /*
     * If salt changes twice between std_crypt calls, complete hashing of
     * pending keys with the old salt.
     */
    if (wq->salt_valid && wq->keys_changed) dispatch_pending("set salt 2", 1);
    wq->salt_valid = 1;
    /*
     * Save salt in work queue structure and decode back username and flags.
     */
    memcpy ( &wq->salt.salt, salt_vp, 12 );
    uaf_packed_convert ( &wq->salt.username, 0 );
    wq->salt.flags = (wq->salt.opt&1) ? UAI$M_PWDMIX : 0;
    VMS_std_crypt_out = crypt_out;	/* make availabe to VMS_FMT module */
    /*
     * Determine if pending set_keys have accumulated enough to cut a work
     * order for worker threads to begin hash operations in parallel.
     */
    pending = (wq->request_count - wq->dispatched);
    if ( !wq->keys_changed ) pending = (-pending);
    DBGPR ( "/VMS_STD/ set salt '%s'(alg=%d), pending keys: %d\n", 
	wq->salt.username.s, wq->salt.alg, pending );

    if ( pending >= wq->pending_threshold ) dispatch_pending ( "set salt", 0 );

}
/*
 * Hash the password and salt saved with VMS_std_set_key and VMS_std_set_salt,
 * saving the result in global storage for retrieval by vms_fmt.c module.
 */
void VMS_std_crypt(void)
{
    DBGPR ( "/VMS_STD/ crypting to complete open work (%d of %d dispatched)\n",
	wq->dispatched, wq->request_count );
    /*
     * Finish any set_key requests pending.  Note that if set_salt changed the
     * salt with no key changes, the entire block of keys will be re-computed.
     */
    dispatch_pending ( "crypt", 1 );
    /*
     * Bump the generation number for the hint.  We use generation number to 
     * mark hint table entries to avoid clearing the entire array each time.
     */
    LOCK_MUTEX ( summary->lock_vp );
    summary->seq = summary->nxtseq;
    summary->nxtseq = (summary->nxtseq+1) & HINT_GENERATION_MASK;
    UNLOCK_MUTEX ( summary->lock_vp );
    /*
     * Keep statistics on crypt() activity.
     */
    wq->crypt_counters.calls++;
    wq->crypt_counters.keys_computed += wq->request_count;
    /*
     * Reset salt_valid and keys_changed so we don't proactively recalculate 
     * until both have been received (or crypt called again after one received).
     */
    wq->salt_valid = 0;
    wq->dispatched = 0;		/* All requests need re-processed */
    wq->assigned = 0;
    wq->keys_changed = 0;	/* defer reseting request_count until change */

    return;
}
/*
 * Return pointer to saved key for position.  Do sanity check with index.
 */
char *VMS_std_get_key ( int position, int index )
{
    if ( wq->request[position].ndx == index ) return wq->request[position].key;

    printf ( "BUGCHECK, wrong index at saved_key position\n" );
    return "";
}
/*
 * Extract salt from ciphertext string to static storage and return
 * pointer to it.  Salt is effectively 70-80 bits (username, salt, 
 * algorithm, pwdmix flag).
 */
char *VMS_std_get_salt(char *ciphertext)
{
    static struct uaf_hash_info pwd;

    uaf_hash_decode ( ciphertext, &pwd );
 
    DBGPR ( "/VMS_STD/ get_salt decoded '%s' to %x/%x-%x-%x-%x-%x\n",
	ciphertext, pwd.salt, pwd.alg, pwd.username.r40[0], pwd.username.r40[1],
	pwd.username.r40[2], pwd.username.r40[3] );
    return (char *) &pwd.salt;
}
/*
 * Extract binary hash from ciphertext into static storage and return
 * pointer to it.
 */
VMS_word *VMS_std_get_binary(char *ciphertext)
{
	static union {
		struct uaf_hash_info pwd;
		VMS_word b[16];
	} out;

	uaf_hash_decode ( ciphertext, &out.pwd );
	DBGPR ( "/VMS_STD/ get_binary decoded '%s' to %lx->%x.%x\n",
		 ciphertext, (unsigned long int) out.b, out.b[1], out.b[0] );

	return out.b;
}
