/*
 * This file is part of John the Ripper password cracker.
 *
 * Converted to OpenVMS format module by David Jones
 *
 * Date:  9-JUL-2009
 * Revised: 25-JUL-2009		Replace long long references with uaf_qword.
 * Revised: 26-AUG-2011		Update for 1.7.8, move pthread.h include.
 * Revised: 19-SEP-2011		Change MIN_KEYS_PER_CRYPT to 1, eliminate
 *				FMT_BS flag.
 *
 * Copyright (c) 2011 by David L. Jones <jonesd/at/columbus.rr.com>, and
 * is hereby released to the general public under the following terms:
 *    Redistribution and use in source and binary forms, with or without
 *    modifications, are permitted.
 */
#include <stdio.h>
#include <string.h>

#include "arch.h"
#include "misc.h"
#include "vms_std.h"
#include "common.h"
#include "formats.h"

#define FORMAT_LABEL			"openvms"
#define FORMAT_NAME			"OpenVMS Purdy"
#define FORMAT_NAME_NOPWDMIX		"OpenVMS Purdy (nopwdmix)"

#define BENCHMARK_COMMENT		" multithreaded"
#define BENCHMARK_LENGTH		-1

#define PLAINTEXT_LENGTH		32
#define CIPHERTEXT_LENGTH		UAF_ENCODE_SIZE

#define BINARY_SIZE			8
#define SALT_SIZE			12

#define MIN_KEYS_PER_CRYPT		1
#define MAX_KEYS_PER_CRYPT		SAVE_LIMIT

static struct fmt_tests tests[] = {
	{"$V$S44zI913bBx-UJrcFSC------D", "President#44"},
	{"$V$9AYXUd5LfDy-aj48Vj54P-----", "USER"},
	{"$V$p1UQjRZKulr-Z25g5lJ-------", "service"},
	{NULL}
};

static int saved_keypos[MAX_KEYS_PER_CRYPT];
static char saved_key[MAX_KEYS_PER_CRYPT][PLAINTEXT_LENGTH + 1];
struct result_hash *VMS_std_crypt_out;
int VMS_dbg_flag;
#define DBGPR if ( VMS_dbg_flag == 0 ); else printf
/*
 * See if signature of ciphertext (from passwd file) matches the hack
 * produced by the uaf_encode routine (starts with $V$)
 */
#if (FMT_MAIN_VERSION > 8)
static int valid(char *ciphertext, struct fmt_main *self )
{
    if (strncmp(ciphertext, "$V$", 3)) return 0;	/* no match */

    if ( strlen ( ciphertext ) < (UAF_ENCODE_SIZE-1) ) return 0;

    return 1;
}
#else
static int valid(char *ciphertext )
{
    if (strncmp(ciphertext, "$V$", 3)) return 0;	/* no match */

    if ( strlen ( ciphertext ) < (UAF_ENCODE_SIZE-1) ) return 0;

    return 1;		/* Only one split */
}
#endif
static void fmt_vms_init ( struct fmt_main *self )
{
printf ( "initializing '%s'\n", self->params.label );
    VMS_std_init();
}
/*
 * Prepare function.
 */
char *prepare ( char *split_fields[10], struct fmt_main *pFmt )
{
    return split_fields[1];
}
/*
 * Hash the internal representation of a ciphertext hash to 4, 8, or 12 bits.
 */
static int binary_hash_0(void *binary)
{
    unsigned short int *bw;
    bw = binary;
    DBGPR ( "/VMS_FMT/ binary_hash_0(%x-%x-%x-%x)\n", bw[0],bw[1],bw[2],bw[3] );
    return ((bw[0]^bw[1]) ^ (bw[2]^bw[3])) & 0x0F;
}

static int binary_hash_1(void *binary)
{
    unsigned short int *bw;
    bw = binary;
    DBGPR ( "/VMS_FMT/ binary_hash_1(%x-%x-%x-%x)\n", bw[0],bw[1],bw[2],bw[3] );
    return ((bw[0]^bw[1]) ^ (bw[2]^bw[3])) & 0x0FF;
}

static int binary_hash_2(void *binary)
{
    unsigned short int *bw;
    bw = binary;
    DBGPR ( "/VMS_FMT/ binary_hash_2(%x-%x-%x-%x)\n", bw[0],bw[1],bw[2],bw[3] );
    return ((bw[0]^bw[1]) ^ (bw[2]^bw[3])) & 0x0FFF;
}
static int binary_hash_3(void *binary)
{
    unsigned short int *bw;
    bw = binary;
    DBGPR ( "/VMS_FMT/ binary_hash_3(%x-%x-%x-%x)\n", bw[0],bw[1],bw[2],bw[3] );
    return ((bw[0]^bw[1]) ^ (bw[2]^bw[3])) & 0x0FFFF;
}
static int binary_hash_4(void *binary)
{
    unsigned short int *bw;
    bw = binary;
    DBGPR ( "/VMS_FMT/ binary_hash_4(%x-%x-%x-%x)\n", bw[0],bw[1],bw[2],bw[3] );
    return ((bw[0]^bw[1]) ^ (bw[2]^bw[3])) & 0x0FFFFF;
}

/*
 * Hash the internal representation of save crypt_all result binary
 * to 4, 8, 12, 16, or 20 bits.

#define PASSWORD_HASH_SIZE_0            0x10
#define PASSWORD_HASH_SIZE_1            0x100
#define PASSWORD_HASH_SIZE_2            0x1000
#define PASSWORD_HASH_SIZE_3            0x10000
#define PASSWORD_HSIZE_4            0x100000
 */
static int get_hash_0(int index)
{
    if ( !VMS_std_crypt_out ) return 0;
    DBGPR ( "/VMS_FMT/ get_hash_0(%d)\n", index );
    return binary_hash_0 ( &VMS_std_crypt_out [index].info.hash );
}

static int get_hash_1(int index)
{
    if ( !VMS_std_crypt_out  ) return 0;
    DBGPR ( "/VMS_FMT/ get_hash_1(%d)\n", index );
    return binary_hash_1 ( &VMS_std_crypt_out [index].info.hash );
}

static int get_hash_2(int index)
{
    if ( !VMS_std_crypt_out  ) return 0;
    DBGPR ( "/VMS_FMT/ get_hash_2(%d)\n", index );
    return binary_hash_2 ( &VMS_std_crypt_out [index].info.hash );
}

static int get_hash_3(int index)
{
    if ( !VMS_std_crypt_out  ) return 0;
    DBGPR ( "/VMS_FMT/ get_hash_3(%d)\n", index );
    return binary_hash_3 ( &VMS_std_crypt_out [index].info.hash );
}

static int get_hash_4(int index)
{
    if ( !VMS_std_crypt_out  ) return 0;
    DBGPR ( "/VMS_FMT/ get_hash_4(%d)\n", index );
    return binary_hash_4 ( &VMS_std_crypt_out [index].info.hash );
}

/*
 * Hash the internal representation of salt (12 bytes) to 10 bits.
 */
static int salt_hash(void *salt)
{
    unsigned short int *bw;
    bw = salt;
    return ((bw[0]^bw[1]) ^ (bw[2]^bw[3]) ^bw[4]) & 0x3FF;
}
/*
 * Save a password (key) for testing.  VMS_std_set_key returns position value
 * we can use if needed to recall the key by a fmt->get_key request.  On get_key
 * return a private copy.
 */
static void set_key(char *key, int index)
{
	saved_keypos[index] = VMS_std_set_key(key, index);
	if ( (*key == 0) && (saved_keypos[index] == 0) ) {
	    /* Special sequence, 'delete' existing key' */
	    VMS_std_crypt();
	}
}

static char *get_key(int index)
{
	char *key = VMS_std_get_key ( saved_keypos[index], index );

	return strcpy ( saved_key[index], key );
}

static int cmp_one(void *binary, int index)
{
    uaf_qword *qw;
    if ( !VMS_std_crypt_out  ) return 0;

    qw = binary;

    DBGPR ("/VMT_FMT/ cmp_one comparing %llx against saved (%llx) [%d]\n",
	*qw, VMS_std_crypt_out [index].info.hash, index );
    return UAF_QW_EQL(*qw, VMS_std_crypt_out [index].info.hash);
}

static int cmp_all(void *binary, int count)
{
    int i;
    uaf_qword *qw;
    uaf_lword *hint, ndx;
    if ( !VMS_std_crypt_out  ) return 0;

    qw = binary;
    /*
     * Check the summary hint.  A possible hit would set the hash table entry
     * to the current generation number so we should probably look for match.
     */
    hint = &VMS_std_hint->seen[UAF_QW_AND(*qw, VMS_std_hint->mask)];
    i = (VMS_std_hint->seq == (*hint & HINT_GENERATION_MASK));
    DBGPR("/VMT_FMT/ cmp_all comparing %llx in saved [0..%d] (%llx): %d\n",
	*qw, count-1, *qw&VMS_std_hint->mask, i );
    if ( !i ) return 0;
    /*
     * Possible hit, extract index and compare if we can.
     */
    ndx = (*hint) >> HINT_GENMASK_BITS;
    if ( ndx > SAVE_LIMIT ) {
	DBGPR ("cmp_all() hint collision, save_limit=%d\n", SAVE_LIMIT );
	return 1;	/* multiple hits, can't single out one to test. */
    }

    return cmp_one ( binary, ndx );
}

static int cmp_exact(char *source, int index)
{
    uaf_qword *qw;

    if ( !VMS_std_crypt_out  ) return 0;
    qw = (uaf_qword *) VMS_std_get_binary(source);
    /*
     * Note that comparison depends upon hash being first member of
     * uaf_hash_info structure.
     */
    return UAF_QW_EQL ( *qw, VMS_std_crypt_out[index].info.hash );
}
static char *fmt_vms_split(char *ciphertext, int index)
{
	return ciphertext;
}
static void fmt_vms_clear_keys(void)
{
}
/*
 * Class record.
 */
struct fmt_main fmt_VMS = {
	{
		FORMAT_LABEL,			/* .label */
		FORMAT_NAME,			/* .format_name */
		VMS_ALGORITHM_NAME,		/* .algorithm_name */
		BENCHMARK_COMMENT,		/* .benchmark_comment */
		BENCHMARK_LENGTH,		/* .benchmark_length (pwd break len) */
		PLAINTEXT_LENGTH,		/* .plaintext_lenght (max) */
		BINARY_SIZE,			/* .binary_size (quadword) */
		SALT_SIZE,			/* .salt_size (word) */
		MIN_KEYS_PER_CRYPT,
		MAX_KEYS_PER_CRYPT,
		FMT_CASE | FMT_8_BIT,		/* .flags */
		tests				/* .tests (sample ciphertexts) */
	}, {
#if (FMT_MAIN_VERSION > 8)
		fmt_vms_init,			/* changed for jumbo */
		prepare,			/* Added for jumbo */
#else
		VMS_std_init,			/* old way */
#endif
		valid,
		fmt_vms_split,
		(void *(*)(char *))VMS_std_get_binary,
		(void *(*)(char *))VMS_std_get_salt,
		{
			binary_hash_0,
			binary_hash_1,
			binary_hash_2,
			binary_hash_3,
			binary_hash_4
		},
		salt_hash,
		(void (*)(void *))VMS_std_set_salt,
		set_key,
		get_key,
		fmt_vms_clear_keys,
		(void (*)(int)) VMS_std_crypt,
		{
			get_hash_0,
			get_hash_1,
			get_hash_2,
			get_hash_3,
			get_hash_4
		},
		cmp_all,
		cmp_one,
		cmp_exact
	}
};
/*
 * Special function that DLL will export, called by --plugin option.
 */
struct fmt_main *FMT_LOADER ( int version )
{
    /*
     * Only allow the load if caller's version matches the one we were
     * compiled with.
     */
    if ( version != FMT_MAIN_VERSION ) {
	fprintf (stderr, 
		"Failed to activate %s format due to version mismatch.\n",
		fmt_VMS.params.label );
	fprintf (stderr,
		"   requested version: %d, supported by DLL: %d\n",
		version, FMT_MAIN_VERSION );
	return 0;
    }
    fmt_VMS.next = 0;
    return &fmt_VMS;
}
