/*
 * This file is part of John the Ripper password cracker.
 *
 * VMS-specific version by David Jones.
 *
 * Date:    18-JUL-2009
 * Revised: 26-JUL-2009		! use UIC to test ownership rather than PID
 *
 * Copyright (c) 2011 by David L. Jones <jonesd/at/columbus.rr.com>, and
 * is hereby released to the general public under the following terms:
 *    Redistribution and use in source and binary forms, with or without
 *    modifications, are permitted.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <starlet.h>		/* VMS system services */
#include <ssdef.h>		/* VMS system service condition codes */
#include <dcdef.h>		/* VMS device class codes */
#include <iodef.h>		/* VMS QIO function codes */
#include <descrip.h>		/* VMS string descriptors */
#include <ttdef.h>		/* Terminal driver definitions */
#include <dvidef.h>		/* VMS get device information codes */
#include <jpidef.h>
#include <lib$routines.h>	/* VMS run-time library functions */
#define EFN 128

#if !defined(O_NONBLOCK) && defined(O_NDELAY)
#define O_NONBLOCK			O_NDELAY
#endif

#include "tty.h"

#ifdef VMS
struct termios {
    long c_lflag;
    long c_cc[2];
};
#define VMIN 0
#define VTIME 1
#define TCSANOW 0
#define ICANON 1
#define ECHO 2
static int tcgetattr(int fd, struct termios *termios_p) 
{
    return 0;
}
static int tcsetattr(int fd, int actions, struct termios *termios_p)
{
    return 0;
}
static long tcgetpgrp(int fd)
{
    return getpid();
}

#endif

static int tty_fd = -1;
static struct termios saved_ti;

struct io_status {
    unsigned short int status, count;
    unsigned short terminator, termlen;
};
/*
 * Buffer to save asychronously read data from terminal (channel tty_fd).
 */
static struct input_buffer {
    struct io_status iosb;
    void (*ast) ( struct input_buffer * );	/* AST for read */
    long terminator_set[2];
    unsigned char line[4];		/* current read */

    int rundown;
    int new_data;
    int interrupt_pending;
    int ctrl_c_pending;

    int data_count;
    int data_read;
    unsigned char data[8192];
} inbuf;

static int queue_input_ast ( struct input_buffer *in )
{
    int status, flags;

    flags = IO$M_NOECHO | IO$M_TRMNOECHO;
    in->terminator_set[0] = in->terminator_set[1] = 0;

    status = SYS$QIO ( EFN, tty_fd, IO$_READVBLK|flags, &in->iosb,
	in->ast, in, in->line, 1, 0, in->terminator_set, 0, 0 );

    return status;
}

static void read_completion ( struct input_buffer *in )
{
    int status, i;
    /*
     * Save input character in data buffer.
     */
    if ( in->rundown ) return;	/* I/O was canceled */
    if ( (in->iosb.status & 1) == 0 ) {
	printf ( "read completion error: %d\n", in->iosb.status );
    } else {
	for ( i = 0; i < (in->iosb.count+in->iosb.termlen); i++ ) {
	    in->data[in->data_count++] = in->line[i];
	}
    }
    status = queue_input_ast ( in );
    /*
     * Force compiler ordering of write by reading buffer that was in QIO.
     */
    if ( in->terminator_set[0] == 0 ) inbuf.new_data = 1;
}

static int enable_ctrl_c ( unsigned short tt_chan, void (*ast), void *arg )
{
    int status;
    struct io_status iosb;

    status = SYS$QIOW ( EFN, tt_chan, IO$_SETCHAR|IO$M_CTRLCAST, &iosb,
		0, 0, ast, arg, 0, 0, 0, 0 );
    if ( (status&1) ) status = iosb.status;
    return status;
}

void ctrl_c_ast ( void *arg_vp )
{
    inbuf.new_data = 1;
    enable_ctrl_c ( tty_fd, ctrl_c_ast, 0 );
    inbuf.ctrl_c_pending++;
}

void tty_init(int stdin_mode)
{
	int status, code, dev_class;
	long uic, tt_owner;
	struct termios ti;
	static $DESCRIPTOR(device_dx,"TT:");
	unsigned short tt_chan;
	struct io_status iosb;

	if (tty_fd >= 0) return;	/* already initialized */
	if ( stdin_mode ) return;	/* reading test words from stdin */
	/*
	 * Assign channel to TT: and verify it is a terminal owned by us.
	 */
	status = SYS$ASSIGN ( &device_dx, &tt_chan, 0, 0, 0 );
	if ( (status&1) == 0 ) {
	    tty_fd = -2;
	    return;
	}
	dev_class = 0;
	code = DVI$_DEVCLASS;
	status = LIB$GETDVI ( &code, &tt_chan, 0, &dev_class, 0, 0 );
	if ( (status&1) == 0 ) dev_class = DC$_MAILBOX;
	if ( dev_class != DC$_TERM ) {
	    SYS$DASSGN ( tt_chan );
	    tty_fd = -3;
	    return;
	}
	code = DVI$_OWNUIC;
	status = LIB$GETDVI ( &code, &tt_chan, 0, &tt_owner, 0, 0 );
	if ( (status&1) == 0 ) tt_owner = -1;
	code = JPI$_UIC;
	status = LIB$GETJPI ( &code, 0, 0, &uic, 0, 0 );
	if ( ((status&1) == 0) || (uic != tt_owner) ) {
	    tty_fd = -4;
	    printf ( "TT not owned by user!\n" );
	    SYS$DASSGN ( tt_chan );
	    return;
	}
	/*
	 * Setup unsolicited input and out-of-band ASTS to catch keyboard
	 * activity and convert to signals.
	 */
	if ( 1 & enable_ctrl_c ( tt_chan, ctrl_c_ast, 0 ) == 0 ) {
	    tty_fd = -4;
	    return;
	}
	tty_fd = tt_chan;

	memset ( &inbuf, 0, sizeof(inbuf) );
	inbuf.ast = read_completion;

	queue_input_ast ( &inbuf );

	atexit(tty_done);
}

int tty_getchar(void)
{
	int c, ctrl_c_pending, interrupt_pending;
	/*
	 * Do quick check for AST or signal handler modifying inbuf.
	 */
	if ((tty_fd >= 0) && inbuf.new_data ) {
	    /*
	     * Disable ASTs and return data.
	     */
	    SYS$SETAST(0);
	    inbuf.new_data = 0;
	    ctrl_c_pending = inbuf.ctrl_c_pending;
	    interrupt_pending = inbuf.interrupt_pending;
	    inbuf.ctrl_c_pending = 0;
	    inbuf.interrupt_pending = 0;
	    if  ( inbuf.data_read < inbuf.data_count ) {
		c = inbuf.data[inbuf.data_read++];
		if ( inbuf.data_read >= 8000 ) {
		    inbuf.data_count -= inbuf.data_read;
		    memcpy ( inbuf.data, &inbuf.data[inbuf.data_read],
			inbuf.data_count );
		    inbuf.data_read = 0;
		}
	    } else {
		c = -1;	/* no data ready */
	    }
	    SYS$SETAST(1);
	    /* if ( ctrl_c_pending ) raise ( SIGHUP ); */
	    return c;
	}

	return -1;
}

void tty_done(void)
{
	unsigned short tt_chan;

	if (tty_fd < 0) return;

	tt_chan = tty_fd; tty_fd = -1;

	inbuf.rundown = 1;	/* kill AST thread */
	SYS$DASSGN ( tt_chan );
}
