/*
 * This file is part of John the Ripper password cracker.
 *
 * Modified for VMS to not interact with pthreads scheduler.
 *
 * Copyright (c) 2011 by David L. Jones <jonesd/at/columbus.rr.com>, and
 * is hereby released to the general public under the following terms:
 *    Redistribution and use in source and binary forms, with or without
 *    modifications, are permitted.
  */

#ifdef __ultrix__
#define __POSIX
#define _POSIX_SOURCE
#endif
#include <pthread.h>
#include <descrip.h>

#ifdef _SCO_C_DIALECT
#include <limits.h>
#endif
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <errno.h>

#ifdef __DJGPP__
#include <dos.h>
#endif

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include "arch.h"
#include "misc.h"
#include "params.h"
#include "tty.h"
#include "config.h"
#include "bench.h"

volatile int event_pending = 0;
volatile int event_abort = 0, event_save = 0, event_status = 0;
volatile int event_ticksafety = 0;

static int timer_save_interval, timer_save_value;
static clock_t timer_ticksafety_interval, timer_ticksafety_value;

#if !OS_TIMER

#include <time.h>
#include <sys/times.h>

static clock_t timer_emu_interval = 0;
static unsigned int timer_emu_count = 0, timer_emu_max = 0;

void sig_timer_emu_init(clock_t interval)
{
	timer_emu_interval = interval;
	timer_emu_count = 0; timer_emu_max = 0;
}

void sig_timer_emu_tick(void)
{
	static clock_t last = 0;
	clock_t current;
	struct tms buf;

	if (++timer_emu_count < timer_emu_max) return;

	current = times(&buf);

	if (!last) {
		last = current;
		return;
	}

	if (current - last < timer_emu_interval && current >= last) {
		timer_emu_max += timer_emu_max + 1;
		return;
	}

	last = current;
	timer_emu_count = 0;
	timer_emu_max >>= 1;

	raise(SIGALRM);
}

#endif

static void sig_install_update(void);

static void sig_handle_update(int signum)
{
	event_save = event_pending = 1;

#ifndef SA_RESTART
	sig_install_update();
#endif
}

static void sig_install_update(void)
{
#ifdef SA_RESTART
	struct sigaction sa;

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = sig_handle_update;
	sa.sa_flags = SA_RESTART;
	sigaction(SIGHUP, &sa, NULL);
#else
	/* signal(SIGHUP, sig_handle_update); */
#endif
}

static void sig_remove_update(void)
{
	signal(SIGHUP, SIG_IGN);
}

void check_abort(int be_async_signal_safe)
{
	if (!event_abort) return;

	if (be_async_signal_safe) {
		write_loop(2, "Session aborted\n", 16);
		_exit(1);
	}

	fprintf(stderr, "Session aborted\n");
	error();
}

static void sig_install_abort(void);

static void sig_handle_abort(int signum)
{
	int saved_errno = errno;

	check_abort(1);

	event_abort = event_pending = 1;

	write_loop(2, "Wait...\r", 8);

	sig_install_abort();

	errno = saved_errno;
}

#ifdef __CYGWIN32__
static CALLBACK BOOL sig_handle_abort_ctrl(DWORD ctrltype)
{
	sig_handle_abort(SIGINT);
	return TRUE;
}
#endif

static void sig_install_abort(void)
{
#ifdef __DJGPP__
	setcbrk(1);
#endif

#ifdef __CYGWIN32__
	SetConsoleCtrlHandler(sig_handle_abort_ctrl, TRUE);
#endif

	signal(SIGINT, sig_handle_abort);
	signal(SIGTERM, sig_handle_abort);
#ifdef SIGXCPU
	signal(SIGXCPU, sig_handle_abort);
#endif
#ifdef SIGXFSZ
	signal(SIGXFSZ, sig_handle_abort);
#endif
}

static void sig_remove_abort(void)
{
#ifdef __CYGWIN32__
	SetConsoleCtrlHandler(sig_handle_abort_ctrl, FALSE);
#endif

	signal(SIGINT, SIG_DFL);
	signal(SIGTERM, SIG_DFL);
#ifdef SIGXCPU
	signal(SIGXCPU, SIG_DFL);
#endif
#ifdef SIGXFSZ
	signal(SIGXFSZ, SIG_DFL);
#endif
}

#ifdef __CYGWIN32__

static int sig_getchar(void)
{
	int c;

	if ((c = tty_getchar()) == 3) {
		sig_handle_abort(CTRL_C_EVENT);
		return -1;
	}

	return c;
}

#else

#define sig_getchar tty_getchar

#endif

static void sig_install_timer(void);

static void sig_handle_timer(int signum)
{
	int saved_errno = errno;

	if (!--timer_save_value) {
		timer_save_value = timer_save_interval;

		event_save = event_pending = 1;
	}

	if (!--timer_ticksafety_value) {
		timer_ticksafety_value = timer_ticksafety_interval;

		event_ticksafety = event_pending = 1;
	}

	if (sig_getchar() >= 0) {
		while (sig_getchar() >= 0);

		event_status = event_pending = 1;
	}

	errno = saved_errno;
}

static struct timer_state {
    pthread_mutex_t lock;
    pthread_cond_t shutdown;

    int running;

    struct timespec interval;

    struct timespec next_tick;

    pthread_t tid;
} interval_ctl = { PTHREAD_MUTEX_INITIALIZER, PTHREAD_COND_INITIALIZER,
	0, { 1, 0 }, { 0, 0 } };
static void *event_monitor ( void *ctl_vp )
{
    int status;
    struct timer_state *ctl;

    ctl = ctl_vp;
    pthread_mutex_lock ( &ctl->lock );
    pthread_get_expiration_np ( &ctl->interval, &ctl->next_tick );
    while ( ctl->running ) {
	status = pthread_cond_timedwait ( &ctl->shutdown, &ctl->lock,
		&ctl->next_tick );
	if ( status == 0 ) continue;	/* check if being shut down */
        /*
	 * Prepare expiration time for next tick.
	 */
	ctl->next_tick.tv_nsec += ctl->interval.tv_nsec;
	if ( ctl->next_tick.tv_nsec > 1000000000 ) {
	    ctl->next_tick.tv_sec++;	/* overflow into seconds field */
	    ctl->next_tick.tv_nsec -= 1000000000;
	}
	ctl->next_tick.tv_sec += ctl->interval.tv_sec;
	/*
	 * Note the event.
	 */
	sig_handle_timer ( SIGALRM );
    }
    /*
     * Someone stopped us from running, let thread exit.
     */
    pthread_mutex_unlock ( &ctl->lock );
    return ctl;
}

static void sig_install_timer(void)
{
    /*
     * Create a small thread that wake periodically to trigger events.
     */
    pthread_attr_t attr;
    pthread_t tid;
    int status;

    pthread_mutex_lock ( &interval_ctl.lock );
    if ( !interval_ctl.running ) {
	pthread_attr_init ( &attr );
	pthread_attr_setdetachstate ( &attr, PTHREAD_CREATE_JOINABLE );
	
	status = pthread_create ( &interval_ctl.tid, &attr, 
		event_monitor, &interval_ctl );
	if ( status == 0 ) {
	    interval_ctl.running = 1;
	} else {
	    fprintf ( stderr, "Error installing timer thread!\n" );
	}
    } else {
	/* Already running */
    }
    pthread_mutex_unlock ( &interval_ctl.lock );
}

static void sig_remove_timer(void)
{
    int was_running;
    void *value;

    pthread_mutex_lock ( &interval_ctl.lock );
    was_running = interval_ctl.running;
    interval_ctl.running = 0;
    pthread_cond_signal ( &interval_ctl.shutdown );
    pthread_mutex_unlock ( &interval_ctl.lock );

    if ( was_running ) pthread_join ( interval_ctl.tid, &value );
}

static void sig_done(void);

void sig_init(void)
{
	clk_tck_init();

	timer_save_interval = cfg_get_int(SECTION_OPTIONS, NULL, "Save");
	if (timer_save_interval < 0)
		timer_save_interval = TIMER_SAVE_DELAY;
	else
	if ((timer_save_interval /= TIMER_INTERVAL) <= 0)
		timer_save_interval = 1;
	timer_save_value = timer_save_interval;

	timer_ticksafety_interval = (clock_t)1 << (sizeof(clock_t) * 8 - 4);
	timer_ticksafety_interval /= clk_tck;
	if ((timer_ticksafety_interval /= TIMER_INTERVAL) <= 0)
		timer_ticksafety_interval = 1;
	timer_ticksafety_value = timer_ticksafety_interval;

	atexit(sig_done);

	sig_install_update();
	sig_install_abort();
	sig_install_timer();
}

static void sig_done(void)
{
	sig_remove_update();
	sig_remove_abort();
	sig_remove_timer();
}
