/*
 * unuaf: convert from OpenVMS sysuaf.dat to /etc/passwd format
 *
 * VMS patch for John the Ripper version 1.6.32
 * (C) 2002 Jean-loup Gailly  http://gailly.net
 * license: GPL <http://www.gnu.org/>
 *
 * This file is based on code from John the Ripper,
 *   Copyright (c) 1996-2002 by Solar Designer.
 * Code from VMSCRACK by Davide Casale and Mario Ambrogetti.
 */

#include <stdio.h>
#include <string.h>

#include "arch.h"
#include "misc.h"
#include "common.h"

static int dump_all = 0; /* set to 1 with -a to dump disabled accounts too */

typedef unsigned char byte;
typedef unsigned short word; /* note:  word in this code is 16 bits */
typedef ARCH_WORD_32 dword;  /* note: dword in this code is 32 bits */

#define USER_SIZE    (32 + 1) /* names are limited to 32 characters */

#if defined(__STDC__)
#  pragma pack(1)
#endif

typedef struct {

  int CTRL_Y        :1;  /* Bit 0 - User can not use CTRL-Y                 */
  int default_com   :1;  /* Bit 1 - User is restricted to default command   */
  int set_pass      :1;  /* Bit 2 - SET PASSWORD command is disabled        */
  int prevent_login :1;  /* Bit 3 - Prevent user from changing any login    */
  int user_account  :1;  /* Bit 4 - User account is disabled                */
  int welcome       :1;  /* Bit 5 - User will not receive the login message */
  int new_mail      :1;  /* Bit 6 - Announcement of new mail is suppressed  */
  int mail_delivery :1;  /* Bit 7 - Mail delivery to user is disabled       */

} L_Flags_0;

typedef struct {

  int gen_passed    :1;  /* Bit 0 - User is required to use generated passwd */
  int primary_pass  :1;  /* Bit 1 - Primary password is expired              */
  int secondary_pass:1;  /* Bit 2 - Secondary password is expired            */
  int act_audited   :1;  /* Bit 3 - All actions are audited                  */
  int last_login    :1;  /* Bit 4 - User will not receive last login messages*/
  int existing_proc :1;  /* Bit 5 - User can not reconnect to existing proces*/
  int user_login    :1;  /* Bit 6 - User can only login to terminals defined */
  int change_pass   :1;  /* Bit 7 - User is required to change expired passwd*/

} L_Flags_1;

typedef struct {

  int captive_acc   :1;   /* Bit 0 - User is restricted to captive account  */
  int user_execut   :1;   /* Bit 1 - Prevent user from executing RUN at DCL */
  int future_use    :6;   /* Bits 2-7 Reserved for future use               */

} L_Flags_2;

typedef struct {

  byte  Record_type;                /* UAF record type */
  byte  Version;                    /* UAF format version */
  word  User_data_offset;           /* Offset of user data counted string   */
  char  Username[32];               /* Username (loginid) space padded      */
  word  Member_UIC;                 /* Mem UIC decimal   1 = 0100           */
                                    /* Mem UIC decimal  10 = 0A00           */
                                    /* Mem UIC decimal 256 = FF01           */
  word  Group_UIC;                  /* Group UIC                            */
  byte  Nulls[12];                  /* Nulls                                */
  char  Account_name[32];           /* Account name                         */
  byte  length_owner;               /* length of owner                      */
  char  Owner[31];                  /* Owner                                */
  byte  length_device;              /* length of device                     */
  char  Device[31];                 /* Device (default disk device)         */
  byte  length_default;             /* length of default (SYS$LOGIN)        */
  char  Default[63];                /* Default (SYS$LOGIN) directory        */
  byte  length_command;             /* length of default login command      */
  char  Default_command[63];        /* Default login command file           */
  byte  length_CLI;                 /* length of default CLI                */
  char  Default_CLI[31];            /* Default CLI                          */
  byte  length_CLI_tables;          /* length of user defined CLI tables    */
  char  User_CLI_table[31];         /* User defined CLI table name          */
  byte  Passwd1[8];                 /* Encrypted primary password           */
  byte  Passwd2[8];                 /* Encrypted secondary password         */
  word  Number_login_fails;         /* Number of login fails                */
  word  Passwd_salt;                /* Password encryption salt             */
  byte  Encr_algo1;                 /* Encryption algorithm code - pass1    */
  byte  Encr_algo2;                 /* Encryption algorithm code - pass2    */
  byte  Pass_minimum_length;        /* Password minimum length              */
  byte  Filler1;                    /* Filler                               */
  byte  Account_ex_date[8];         /* Account expiration date              */
  byte  Password_lifetime[8];       /* Password lifetime                    */
  byte  Pass1_change_date[8];       /* Primary password change date/time    */
  byte  Pass2_change_date[8];       /* Secondary password change date/time  */
  byte  Last_interactive[8];        /* Last interactive login date/time     */
  byte  Last_non_interactive[8];    /* Last non-interactive login date/time */
  byte  Authorize_priviledges[8];   /* Authorize priviledges                */
  byte  Default_priviledges[8];     /* Default priviledges                  */
  byte  Filler2[40];                /* Filler                               */
  byte  Login_Flags_0;              /* Login Flags bits byte 0              */
  byte  Login_Flags_1;              /* Login Flags bits byte 1              */
  byte  Login_Flags_2;              /* Login Flags bits byte 2              */
  byte  Login_Flags_3;              /* Login Flags bits byte 3              */
  byte  Network_primary_days[3];    /* Network access bytes - primary days  */
  byte  Network_seconday_days[3];   /* Network access bytes - seconday days */
  byte  Batch_primary_days[3];      /* Batch access bytes - primary days    */
  byte  Batch_seconday_days[3];     /* Batch access bytes - seconday days   */
  byte  Local_primary_days[3];      /* Local access bytes - primary days    */
  byte  Local_seconday_days[3];     /* Local access bytes - seconday days   */
  byte  Dialup_primary_days[3];     /* Dialup access bytes - primary days   */
  byte  Dialup_secondary_days[3];   /* Dialup access bytes - secondary days */
  byte  Remote_primary_days[3];     /* Remote access bytes - primary days   */
  byte  Remote_seconday_days[3];    /* Remote access bytes - seconday days  */
  byte  Filler3[12];                /* Filler                               */
  byte  Prime_days;                 /* Prime days, offset=514=0x202         */
  byte  Filler4;                    /* Filler                               */
  byte  Default_base_priority;      /* Default base priority                */
  byte  Maximum_job_queue_priority; /* Maximum job queue priority           */
  word  Active_process_limit;       /* Active process limit, off=518=0x206  */
  word  Max_number_jobs;            /* Max. number of interactive jobs      */
  word  Detached_process_limit;     /* Detached process limit               */
  word  Subprocess_creation_limit;  /* Subprocess creation limit            */
  word  Buffered_IO_limit;          /* Buffered I/O limit                   */
  word  Direct_IO_limit;            /* Direct I/O limit, offset 528=0x210   */
  word  Timer_queue_entry_limit;    /* Timer queue entry limit              */
  word  AST_queue_limit;            /* AST queue limit                      */
  word  Lock_queue_limit;           /* Lock queue limit                     */
  word  Open_file_limit;            /* Open file limit                      */
  word  Shared_file_limit;          /* Shared file limit                    */
  dword Working_set_quota;          /* Working set quota                    */
  dword Working_set_count;          /* Working set count, offset 544=0x220  */
  dword Working_set_extent;         /* Working set extent                   */
  dword Paging_file_quota;          /* Paging file quota                    */
  dword Maximum_CPU_time_limit;     /* Maximum CPU time limit (in 10-msec)  */
  dword Buffered_IO_byte_limit;     /* Buff. I/O byte limit, offs 560=0x230 */
  dword Paged_buffer_IO_count_limit;/* Paged buffer I/O byte count limit    */
  dword Job_log_name_table_quota;   /* logical name table creation quota    */
  byte  Filler5[72];                /* Filler. Most records have 644 bytes  */

} UAF;

#define MAX_UAF_RECSIZE 1412 /* Some uaf records have 1412 bytes */


#if ARCH_LITTLE_ENDIAN
#  define hostWord(w) (w)
#else
   /* convert word read from memory in little-endian format to host format: */
#  define hostWord(w) (((w)>>8) + (((w) & 0xff)<<8))
#endif

/* ============================================================================
 * Remove trailing spaces.
 * s is limited to 32 characters and is NOT zero terminated.
 * The return value is zero terminated.
 */
static char *strip_spaces(char *s)
{
    static char name[USER_SIZE];
    int n;

    for (n = 0; n < USER_SIZE && s[n] != ' '; n++) {
	name[n] = s[n];
    }
    name[n] = '\0';
    return name;
}

/* ============================================================================
 * Read the sysuaf file and dump it in /etc/password format
 */
static void process_uaf(FILE *file)
{
    UAF  user;
    int n;
    word salt;

    while (fread(&user, sizeof(user), 1, file) == 1) {

	/* If the previous record had 1412 bytes, and we have just read the
         * middle 644 bytes, skip until the record end. The test is not
         * reliable but I don't known another portable way of doing it on
         * non-VMS systems (when sysuaf.dat has been transfered first).
         */
	if (user.Record_type > 1 || user.Version > 1 ||
	    user.Username[31] != ' ') {
	    if (fread(&user, 1, MAX_UAF_RECSIZE-2*sizeof(user), file) <= 0) {
		pexit("unexpected eof");
	    }
	    continue;
	}
        if (!dump_all && ((L_Flags_0 *)(&user.Login_Flags_0))->user_account) {
	    continue;
	}
        /* dump with format:
         * userid:$VMSn$salthashuserid:uid:gid:name:homedir:cmdfile
         */
	salt = hostWord(user.Passwd_salt);
	printf("%s:$VMS%d$%02X%02X",
	       strip_spaces(user.Username),
	       user.Encr_algo1, 
               salt & 0xFF,
               salt >> 8);

	for (n = 0; n < sizeof(user.Passwd1); n++) {
	    printf("%02X", user.Passwd1[n]);
	}
	printf("%s:%d:%d:",
	       strip_spaces(user.Username), /* needed as additional salt */
	       hostWord(user.Member_UIC),
	       hostWord(user.Group_UIC));
	printf("%s - %.*s:%.*s:%.*s\n",
	       strip_spaces(user.Account_name),
	       (int)user.length_owner, user.Owner,
	       (int)user.length_default, user.Default,
	       (int)user.length_command, user.Default_command);
    }
}

/* ============================================================================
 * Open the sysuaf file and dump it in /etc/password format.
 * Use the -a option to dump all users including disabled accounts.
 */
int unuaf(int argc, char **argv)
{
    FILE *file;
    char *sysuaf;

    if (argc >= 2 && !strcmp(argv[1], "-a")) {
	dump_all = 1;
	argc--, argv++;
    }
    if (argc < 1 || argc > 2) {
	printf("Usage: %s [-a] [sysuaf.dat]\n", argv[0] ? argv[0] : "unuaf");
	return 0;
    }
    sysuaf = argc == 2 ? argv[1] : "sysuaf.dat";

    if (!(file = fopen(sysuaf, "rb"))) {
	pexit("fopen: %s", sysuaf);
    }
    process_uaf(file);

    if (ferror(file)) pexit("fread");

    if (fclose(file)) pexit("fclose");

    return 0;
}
