##################################################################
#  Filename: netscreen.py
#
# This script will generate a netscreen formatted password
#
# This program requires two commandline arguments, and works in two modes:
#   Mode 1:
#    The first argument is a username
#    The second argument is a plaintext password
#   Mode 2:
#    The first argument is -f to indicate reading usernames and passwords from a file
#    The second argument is the filename to read
#
#  The input file should have the following format:
#    <username>,<plain-text-password>
#
#   Example input file:
#     admin,netscreen
#     cisco,cisco
#     robert,harris
#
# Output will be the username and hashed password in John the Ripper format  
# If reading usernames and passwords from a file, the output file name will be: netscreen-JtR-output.txt  
#
#  Version  2.0  
#  Updated on August 12, 2010 by Robert B. Harris from VA 
#    Updated to use the hashlib library
#    Updated to print help text if both input arguments are missing
#    Updated to optionally read from a file
#
##################################################################

import hashlib
import sys

def net(user, password):
  b64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"
  middle = "Administration Tools"
  s = "%s:%s:%s" % (user, middle, password)
  m = hashlib.md5(s).digest()

  narray = []
  for i in range(8):
    n1 = ord(m[2*i])
    n2 = ord(m[2*i+1])
    narray.append( (n1<<8 & 0xff00) | (n2 & 0xff) )

  res = ""
  for i in narray:
    p1 = i >> 12 & 0xf
    p2 = i >> 6  & 0x3f
    p3 = i       & 0x3f
    res = res + b64[p1] + b64[p2] + b64[p3]

  for c, n in  zip("nrcstn", [0, 6, 12, 17, 23, 29]):
	  res = res[:n] + c + res[n:]
  return res


if __name__ == '__main__':
  if len(sys.argv) == 3:
    if  sys.argv[1]== "-f":  # If true, reading from a file
       in_file = sys.argv[2]   # 2nd commandline arg is the filename to read from
       input_file = open( in_file, 'r')
       output_file = open ("netscreen-JtR-output.txt" , 'w')
       for line in input_file:
          data=line.strip('\n').split(',')
          username = data[0]
          password = data[1]
          ciphertext = net(username,password)
          output_file.write ("%s:%s$%s" % (username,username,ciphertext))
	  output_file.write ("\n")
       input_file.close()
       print "\nThe output file has been created."
       output_file.close()
    else:   # We are not reading from a file
      username = sys.argv[1]
      password = sys.argv[2]
      ciphertext = net(username,password)
      print "%s:%s$%s" % (username,username,ciphertext)
  else:   # User did not input the required two commandline arguments
    print "\n\n" 
    print "This program requires two commandline arguments:"
    print "The first argument is a username."
    print "The second argument is a plaintext password.\n"
    print "Output will be the username and hashed password in John the Ripper format.\n\n"
    print "Example"
    print " Input: netscreen.py admin netscreen"
    print "Output: admin:admin$nKv3LvrdAVtOcE5EcsGIpYBtniNbUn"
    print "(Netscreen uses the username as the salt)" 
