#!/usr/bin/perl
#
#####################################################################
#   JoMo-Kun <jmk@foofus.net>
#
#   Crack LM/NTLM response using LM password case-insensitive seed.
#
#   Fix character case for cracked LM responses using respective 
#   NTLM response.
#
#####################################################################
#

use Getopt::Long;

my $VERSION = "0.1";
my %opt;
my %data;

$potfile = 'john.pot';
$JOHN = 'john';

GetOptions (
  'seed=s'      => \$opt{'seed'},
  'file=s'      => \$opt{'file'},
  'help|h'      => sub { ShowUsage(); },
);

sub showUsage {
  print "john-netntlm.pl V. $VERSION\n";
  print "JoMo-Kun <jmk@foofus.net>\n\n";
  print "Usage: $0 [OPTIONS]\n";
  print " $0\n";
  print "   --seed [RainbowCrack/HalfLM Response Password]\n";
  print "   --file [File Containing LM/NTLM challenge/responses (.lc format)]\n";
  print "          Ex: Domain\\User:::LM response:NTLM response:challenge";
  print "\n";
  print " Ex:\n";
  print " $0 --file capture.lc\n";
  print " $0 --seed \"GERGE!!\"--file capture.lc\n";
  print "\n";
  exit(1);
}

# MAIN
{
  if ( !defined($opt{'file'}) ) { &showUsage; }

  # Parse accounts to audit
  open(HAND, $opt{'file'}) || die("Failed to open response file: $opt{'file'} -- $!");
  @{ $data{'pairs'} } = <HAND>;
  close(HAND);

  # Load information for any accounts previous cracked  
  print STDERR "\n\n";
  print STDERR "###########################################################################################\n";  
  
  open (HAND, "$JOHN -format:netlm -show $opt{'file'} |") || die("Failed to execute john: $!");
  print STDERR "The following LM responses have been previously cracked:\n";
  while(<HAND>) {
    next if ( /\d+ password hashes cracked, \d+ left/ );
    last if /^$/;
    print "\t$_";
    push @{ $data{'cracked-lm'} }, $_;
  }
  close(HAND);

  print STDERR "\nThe following NTLM responses have been previously cracked:\n";
  open (HAND, "$JOHN -format:netntlm -show $opt{'file'} |") || die("Failed to execute john: $!");
  while(<HAND>) {
    next if ( /\d+ password hashes cracked, \d+ left/ );
    last if /^$/;
    print "\t$_";
    push @{ $data{'cracked-ntlm'} }, $_;
  }
  close(HAND);

  my $tmpconf = &createConf();
  my $tmpsession = "john.session.$$";
  my $tmpsessionlog = "john.session.log"; # should we unlink this when done?
  #print STDERR "Created temporary configuration file: $tmpconf\n";

  # Crack case-sensitive version of password
  my $tmpdict = "john.dict.$$";
  #print STDERR "Created temporary dictionary file: $tmpdict\n";

  foreach $credential_set ( @{ $data{'cracked-lm'} } ) {
    my ($account,$lmpass,$bar,$netlm,$netntlm,$chall) = split(/:/, $credential_set);
    next if ( grep(/^$account:/i, @{ $data{'cracked-ntlm'} }) );
    
    print STDERR "\n\n";
    print STDERR "###########################################################################################\n";  
    print STDERR "Performing NTLM case-sensitive crack for account: $account.\n"; 

    open(HAND, ">$tmpdict") || die("Failed to option file: $tmpdict -- $!");
    print HAND "$lmpass";
    close(HAND);

    open (HAND, "$JOHN -format:netntlm -config:$tmpconf -wordlist:$tmpdict -rules -user:\"$account\" -session:$tmpsession $opt{'file'} |") || die("Failed to execute john: $!");
    while(<HAND>) { print; }
    close(HAND);
    unlink $tmpdict;   
  }

  print STDERR "\n\n";
  print STDERR "###########################################################################################\n";  
  print STDERR "Isolating accounts which have only had their LM response cracked.\n"; 
  
  foreach $credential_set ( @{ $data{'pairs'} } ) {
    $credential_set =~ s/\\/\\\\/g;
    my ($account,$foo,$bar,$netlm,$netntlm,$chall) = split(/:/, $credential_set);
    if (lc($netlm) eq lc($netntlm)) {
      print STDERR "LM response is not unique from NTLM response (skipping):\n\t$credential_set\n";
      push  @{ $data{'pairs-ntlm'} }, $credential_set;
    }
    elsif ( @cracked = grep(/^$account:/i, @{ $data{'cracked-ntlm'} }) ) {
      print STDERR "Account $account NTLM response previously cracked.\n";
      #print "@cracked";
    }
    else {
      print STDERR "Account $account LM response added to cracking list.\n";
      push  @{ $data{'pairs-lm'} }, $credential_set;
    }
  }

  if ( defined($opt{'seed'}) ) {
    print STDERR "\n\n";
    print STDERR "###########################################################################################\n";  
    print STDERR "Testing seed password to determine whether it is the actual password.\n";
    open(HAND, ">$tmpdict") || die("Failed to option file: $tmpdict -- $!");
    print HAND $opt{'seed'};
    close(HAND);
    
    open (HAND, "$JOHN -format:netntlm -config:$tmpconf -wordlist:$tmpdict -rules -session:$tmpsession $opt{'file'} |") || die("Failed to execute john: $!");
    while(<HAND>) { print; }
    close(HAND);
    unlink $tmpdict;   
 
    my $tmppasswd = "john.passwd.$$";
    open(HAND, ">$tmppasswd") || die("Failed to open $tmppasswd: $!");
    print HAND  @{ $data{'pairs-lm'} };
    close(HAND);

    print STDERR "\n\n";
    print STDERR "###########################################################################################\n";  
    print STDERR "The hashes contained within $tmppasswd have not been cracked.\n";
    print STDERR "Executing the following (this could take a while...):\n\n";
    print STDERR "john -format:netlm -config:$tmpconf -external:HalfLM -incremental:LM -session:$tmpsession $tmppasswd\n";
    print STDERR "\n";
    print STDERR " *If the passwords successfully crack, use this script again to crack the case-sensitive password\n";
    print STDERR " without feeding a seed password\n";
    print STDERR"\n\n";
  
    exec("$JOHN -format:netlm -config:$tmpconf -external:HalfLM -incremental:LM -session:$tmpsession $tmppasswd");
  }

  unlink $tmppasswd, $tmpconf, $tmpsession;
}

exit(0);

sub createConf {
  my $tmpconf = "john.conf.$$";
  open(CONF, ">$tmpconf") || die("Failed to open $tmpconf: $!");

  # Define character keyspace
  print CONF "[Incremental:LM]\n";
  print CONF "File = /usr/share/john/lanman.chr\n";
  print CONF "MinLen = 1\n";
  
  # John compiled for MaxLen <= 8
  if (14 - length($opt{'seed'}) > 8) {
    print CONF "MaxLen = 8\n";
  } else {
    print CONF "MaxLen = ", 14 - length($opt{'seed'}), "\n";
  }
  print CONF "CharCount = 69\n\n";

  # Add external filter to handle uncracked characters
  if ($opt{'seed'} ne "") {
    my $i; $j;
    my @seed = split(//, $opt{'seed'});
   
    print CONF "[List.External:HalfLM]\n";
    print CONF "void filter()\n";
    print CONF "{\n";
  
    my $len = length($opt{'seed'}); 
    for ($i = 13, $j = 13 - $len; $i>=0; $i--) {
      if ($i >= $len) {
        print CONF "  word[$i] = word[$j];\n";
        $j--; 
      } else {
        print CONF "  word[$i] = \'$seed[$i]\';\n";
      }
    }

    print CONF "}\n\n";  
  } 
  
  # Add custom wordlist to utilize NTLM hash for character case cracking
  print CONF "[List.Rules:Wordlist]\n";
  print CONF "l\n";
  print CONF "lMT[*0]T[*1]T[*2]T[*3]T[*4]T[*5]T[*6]T[*7]T[*8]T[*9]T[*A]T[*B]T[*C]T[*D]Q\n";
  
  close(CONF);

  return $tmpconf;
}
